// Фоновой сервис-воркер MV3
// Здесь можно слушать события клика по иконке, сообщений из content.js и т.д.
chrome.runtime.onInstalled.addListener((details) => {
	console.log('[nFGR/SW] Installed', { reason: details?.reason });
});

chrome.runtime.onStartup?.addListener(() => {
	console.log('[nFGR/SW] Startup');
});

chrome.action.onClicked?.addListener(async (tab) => {
	try {
		if (!tab?.id) return;
		console.log('[nFGR/SW] Action clicked, injecting content.js into tab', { tabId: tab.id, url: tab.url });
		await chrome.scripting.executeScript({
			target: { tabId: tab.id },
			files: ['content.js']
		});
	} catch (err) {
		console.error('[nFGR/SW] Inject content.js failed', err);
	}
});

chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
	console.log('[nFGR/SW] onMessage', {
		type: message?.type,
		fromTabId: sender?.tab?.id,
		fromUrl: sender?.url
	});
	// Маршрут: получить заголовок активной вкладки
	if (message?.type === 'GET_TITLE') {
		(async () => {
			try {
				console.log('[nFGR/SW] GET_TITLE: querying active tab');
				const [tab] = await chrome.tabs.query({ active: true, lastFocusedWindow: true });
				if (!tab?.id) {
					console.warn('[nFGR/SW] GET_TITLE: no active tab found');
					sendResponse({ ok: false, error: 'NO_ACTIVE_TAB' });
					return;
				}
				const resp = await chrome.tabs.sendMessage(tab.id, { type: 'GET_PAGE_TITLE' });
				console.log('[nFGR/SW] GET_TITLE: response from content', resp);
				sendResponse({ ok: true, ...resp });
			} catch (error) {
				console.error('[nFGR/SW] GET_TITLE error', error);
				sendResponse({ ok: false, error: String(error) });
			}
		})();
		// вернуть true, чтобы оставить канал открытым для асинхронного ответа
		return true;
	}

	// Обработчик: получение ответа по questionId с внешнего ресурса и запись в storage
	if (message?.type === 'FETCH_ANSWER') {
		(async () => {
			/** @type {string} */
			const questionId = String(message?.questionId ?? '');
			if (!questionId) {
				console.warn('[nFGR/SW] FETCH_ANSWER: empty questionId');
				sendResponse({ ok: false, error: 'EMPTY_QUESTION_ID' });
				return;
			}

			const endpoint = `https://62.113.103.245.nip.io/questions/${encodeURIComponent(questionId)}/answer-id`;
			console.log('[nFGR/SW] FETCH_ANSWER: start fetch', { questionId, endpoint });
			/** @type {{ questionId: string, answerInternalId: string|null, status: 'ok'|'not-found'|'error', fetchedAt: number }} */
			let result = {
				questionId,
				answerInternalId: null,
				status: 'error',
				fetchedAt: Date.now()
			};

			try {
				const resp = await fetch(endpoint, { method: 'GET' });
				console.log('[nFGR/SW] FETCH_ANSWER: http status', { status: resp.status, ok: resp.ok });
				if (resp.status === 404) {
					result.status = 'not-found';
				} else if (resp.ok) {
					let data = null;
					try {
						data = await resp.json();
					} catch (e) {
						// no-op
					}
					console.log('[nFGR/SW] FETCH_ANSWER: json parsed', data);
					const answerId = data?.answer_internal_id != null ? String(data.answer_internal_id) : null;
					if (answerId != null) {
						result.answerInternalId = answerId;
						result.status = 'ok';
					} else {
						result.status = 'error';
					}
				} else {
					result.status = 'error';
				}

				// Сохраняем в chrome.storage.local: nFGR.answersByQuestionId[questionId] = result
				const current = await storageGet(['nFGR']);
				const nFGR = current?.nFGR && typeof current.nFGR === 'object' ? current.nFGR : {};
				if (!nFGR.answersByQuestionId || typeof nFGR.answersByQuestionId !== 'object') {
					nFGR.answersByQuestionId = {};
				}
				nFGR.answersByQuestionId[questionId] = result;
				console.log('[nFGR/SW] FETCH_ANSWER: writing to storage', {
					key: `nFGR.answersByQuestionId.${questionId}`,
					value: result
				});
				await storageSet({ nFGR });

				console.log('[nFGR/SW] FETCH_ANSWER: stored', result);
				sendResponse({ ok: true, result });
			} catch (error) {
				console.error('[nFGR/SW] FETCH_ANSWER error', error);
				sendResponse({ ok: false, error: String(error) });
			}
		})();
		// async
		return true;
	}

	// Пример: проксирование запросов к внешнему API из content.js
	// sendResponse({ ok: true })
	sendResponse({ ok: true, echo: message });
	// вернуть true, если ответ асинхронный
	return false;
});

// Логирование изменений в chrome.storage.local (особенно ключ nFGR)
chrome.storage.onChanged.addListener((changes, areaName) => {
	if (areaName !== 'local') return;
	if (changes?.nFGR) {
		console.log('[nFGR/SW] storage.local nFGR changed', {
			oldValue: changes.nFGR.oldValue,
			newValue: changes.nFGR.newValue
		});
	}
});

// --- helpers ---
function storageGet(keys) {
	return new Promise((resolve) => {
		try {
			chrome.storage.local.get(keys, (res) => resolve(res));
		} catch (e) {
			console.error('[nFGR/SW] storageGet error', e);
			resolve({});
		}
	});
}

function storageSet(obj) {
	return new Promise((resolve) => {
		try {
			chrome.storage.local.set(obj, () => resolve());
		} catch (e) {
			console.error('[nFGR/SW] storageSet error', e);
			resolve();
		}
	});
}


