// Content-script: выполняется на страницах (по правилам matches из manifest.json)
console.log('Masha Site Parser: content script loaded');

// Пример парсинга DOM и отправки данных в SW/бэкэнд
async function collectAndSend() {
    const title = getPageTitleFromHtml();
    const url = location.href;
    const text = document.body?.innerText?.slice(0, 2000) || '';

    // Отправка в service worker (далее можно звать внешний API)
    const response = await chrome.runtime.sendMessage({
        type: 'PAGE_DATA',
        payload: { title, url, text }
    });
    console.log('SW response:', response);
}

collectAndSend().catch(console.error);

// Обработчик: вернуть заголовок/URL по запросу из background/popup
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    if (message?.type === 'GET_PAGE_TITLE') {
        sendResponse({ title: getPageTitleFromHtml(), url: location.href });
        return;
    }
});

function getPageTitleFromHtml() {
    // 1) <title>
    const titleEl = document.querySelector('head > title');
    const headTitle = titleEl?.textContent?.trim();
    if (headTitle) return headTitle;

    // 2) <meta property="og:title">
    const og = document.querySelector('meta[property="og:title"]')?.getAttribute('content')?.trim();
    if (og) return og;

    // 3) <meta name="twitter:title">
    const tw = document.querySelector('meta[name="twitter:title"]')?.getAttribute('content')?.trim();
    if (tw) return tw;

    // 4) Первый <h1>
    const h1 = document.querySelector('h1')?.textContent?.trim();
    if (h1) return h1;

    // 5) Fallback: document.title (может быть изменён скриптами), либо хост
    return document.title || location.hostname;
}

// -------------------- in.3level.ru: пуллинг и парсинг страницы тестирования --------------------
/**
 * @typedef {{ url: string, questionId: string, answerIds: string[], timestamp: number }} Snapshot
 * @typedef {{ questionId: string, answerInternalId: string|null, status: 'ok'|'not-found'|'error', fetchedAt: number }} AnswerFetchResult
 */
const ACTIVATE_LETTER_DEFAULT = 'N';

(function bootstrapPolling() {
	try {
		if (!location.hostname.endsWith('in.3level.ru')) {
			return;
		}
		console.log('[nFGR] Polling enabled for in.3level.ru');
		const POLL_MS = 5000;
		// Сразу тик + каждые 5 сек
		tick().catch(console.error);
		setInterval(() => tick().catch(console.error), POLL_MS);
	} catch (e) {
		console.error('[nFGR] bootstrapPolling error', e);
	}
})();

async function tick() {
	if (!isTestingPage()) {
		return;
	}
	const questionId = getQuestionIdFromDOM();
	if (!questionId) {
		return;
	}
	const answerIds = getAnswerIdsFromDOM();

	// Проверяем последний обработанный вопрос
	const { nFGR } = await storageGet(['nFGR']);
	const lastProcessed = nFGR?.lastProcessedQuestionId ? String(nFGR.lastProcessedQuestionId) : null;

	if (lastProcessed !== questionId) {
		console.log('[nFGR] New question detected', { questionId, answerIds });
		// Сохраняем снапшот и lastProcessedQuestionId
		const snapshot = /** @type {Snapshot} */({
			url: location.href,
			questionId,
			answerIds,
			timestamp: Date.now()
		});
		const next = typeof nFGR === 'object' && nFGR ? nFGR : {};
		next.lastSnapshot = snapshot;
		next.lastProcessedQuestionId = questionId;
		await storageSet({ nFGR: next });

		// Просим background получить ответ
		try {
			const resp = await chrome.runtime.sendMessage({ type: 'FETCH_ANSWER', questionId });
			console.log('[nFGR] FETCH_ANSWER resp', resp);
		} catch (e) {
			console.error('[nFGR] FETCH_ANSWER failed', e);
		}
	}
}

function isTestingPage() {
	const form = document.querySelector('form[name="test_form"]');
	const hidden = document.querySelector('form[name="test_form"] input[name="current_question"]');
	return !!(form && hidden);
}

function getQuestionIdFromDOM() {
	const el = document.querySelector('form[name="test_form"] input[name="current_question"]');
	const val = el?.getAttribute('value') || el?.value || '';
	return val ? String(val).trim() : null;
}

function getAnswerIdsFromDOM() {
	const nodes = document.querySelectorAll('input[name="answer"][type="radio"]');
	const ids = Array.from(nodes)
		.map((el) => el.getAttribute('value'))
		.filter((v) => v != null)
		.map((v) => String(v))
		.filter((v, i, a) => a.indexOf(v) === i);
	return ids;
}

// -------------------- Горячая клавиша activate_letter --------------------
(function bootstrapHotkey() {
	try {
		if (!location.hostname.endsWith('in.3level.ru')) return;
		window.addEventListener('keydown', async (e) => {
			try {
				// игнор, если печатаем в полях ввода
				const t = /** @type {any} */(e.target);
				const tag = (t?.tagName || '').toLowerCase();
				if (tag === 'input' || tag === 'textarea' || t?.isContentEditable) return;

				const expectedLetter = (await getActivateLetter())?.toUpperCase?.() || ACTIVATE_LETTER_DEFAULT;
				// сравниваем по физическому коду клавиши (независимо от раскладки)
				const expectedCode = letterToCode(expectedLetter);
				const pressedCode = String(e.code || '');
				if (expectedCode) {
					if (pressedCode !== expectedCode) return;
				} else {
					// fallback для нестандартных символов
					const pressed = String(e.key || '').toUpperCase();
					if (pressed !== expectedLetter) return;
				}

				if (!isTestingPage()) return;
				const questionId = getQuestionIdFromDOM();
				if (!questionId) return;

				// достаём ответ из общего стораджа
				const { nFGR } = await storageGet(['nFGR']);
				const answerInternalId = nFGR?.answersByQuestionId?.[questionId]?.answerInternalId ?? null;
				if (answerInternalId == null) {
					console.warn('[nFGR] Hotkey: no answer for question', { questionId });
					return;
				}
				const selector = `input[name="answer"][type="radio"][value="${CSS.escape(String(answerInternalId))}"]`;
				const radio = document.querySelector(selector);
				if (radio instanceof HTMLInputElement) {
					radio.click();
					// на всякий случай продублируем change для сайтов, которые слушают только его
					radio.dispatchEvent(new Event('change', { bubbles: true }));
					console.log('[nFGR] Hotkey: selected answer', { questionId, answerInternalId });
				} else {
					console.warn('[nFGR] Hotkey: radio not found', { selector });
				}
			} catch (err) {
				console.error('[nFGR] Hotkey error', err);
			}
		}, true);
		console.log('[nFGR] Hotkey enabled');
	} catch (e) {
		console.error('[nFGR] bootstrapHotkey error', e);
	}
})();

async function getActivateLetter() {
	try {
		const { nFGR } = await storageGet(['nFGR']);
		const letter = nFGR?.settings?.activateLetter;
		if (typeof letter === 'string' && letter) return letter;
	} catch { /* ignore */ }
	return ACTIVATE_LETTER_DEFAULT;
}

function letterToCode(ch) {
	try {
		const u = String(ch || '').toUpperCase();
		if (u.length !== 1) return null;
		if (u >= 'A' && u <= 'Z') return 'Key' + u;
		if (u >= '0' && u <= '9') return 'Digit' + u;
		if (u === ' ') return 'Space';
	} catch { /* ignore */ }
	return null;
}

// --- storage helpers (promisified) ---
function storageGet(keys) {
	return new Promise((resolve) => {
		try {
			chrome.storage.local.get(keys, (res) => resolve(res));
		} catch (e) {
			resolve({});
		}
	});
}

function storageSet(obj) {
	return new Promise((resolve) => {
		try {
			chrome.storage.local.set(obj, () => resolve());
		} catch (e) {
			resolve();
		}
	});
}